<?php
function randomIpByCountry(string $iso, string $dir = __DIR__ . '/ranges'): ?string {
  $file = $dir . '/' . strtoupper($iso) . '.json';
  if (!is_file($file)) {
    error_log("Файл {$file} не найден");
    return null;
  }

  $ranges = json_decode(file_get_contents($file), true);
  if (!is_array($ranges) || !$ranges) {
    error_log("Пустой или невалидный JSON в {$file}");
    return null;
  }

  $range = $ranges[array_rand($ranges)];
  if (!is_array($range) || count($range) < 2) {
    error_log("Некорректный диапазон в {$file}");
    return null;
  }

  [$start, $end] = $range;
  $startInt = ip4ToInt($start);
  $endInt   = ip4ToInt($end);

  if ($startInt === null || $endInt === null || $startInt > $endInt) {
    error_log("Ошибка конвертации IP в {$file}");
    return null;
  }

  $randInt = random_int($startInt, $endInt);
  return intToIp4($randInt);
}

function ip4ToInt(string $ip): ?int {
  $parts = array_map('intval', explode('.', $ip));
  if (count($parts) !== 4) return null;
  foreach ($parts as $p) if ($p < 0 || $p > 255) return null;

  return ($parts[0] << 24) | ($parts[1] << 16) | ($parts[2] << 8) | $parts[3];
}

function intToIp4(int $int): string {
  return join('.', [
    ($int >> 24) & 0xFF,
    ($int >> 16) & 0xFF,
    ($int >> 8)  & 0xFF,
    $int & 0xFF
  ]);
}

/* ------------ пример использования ------------ */

echo randomIpByCountry('PL') . PHP_EOL;

function getCountryMeta($phone) {
    // Массив с данными стран
    $countries = [
        ['c' => '+998', 'n' => 'Uzbekistan', 'iso' => 'UZ', 'ip' => '188.113.192.0'],
        ['c' => '+997', 'n' => 'Kazakhstan', 'iso' => 'KZ', 'ip' => '164.0.255.0'],
        ['c' => '+996', 'n' => 'Kyrgyzstan', 'iso' => 'KG', 'ip' => '158.181.255.0'],
        ['c' => '+995', 'n' => 'Georgia', 'iso' => 'GE', 'ip' => '37.156.175.0'],
        ['c' => '+994', 'n' => 'Azerbaijan', 'iso' => 'AZ', 'ip' => '188.253.255.0'],
        ['c' => '+992', 'n' => 'Tajikistan', 'iso' => 'TJ', 'ip' => '217.11.177.0'],
        ['c' => '+974', 'n' => 'Qatar', 'iso' => 'QA', 'ip' => '37.211.64.0'],
        ['c' => '+972', 'n' => 'Israel', 'iso' => 'IL', 'ip' => '109.67.255.0'],
        ['c' => '+971', 'n' => 'United Arab Emirates', 'iso' => 'AE', 'ip' => '151.239.4.209'],
        ['c' => '+90', 'n' => 'Turkey', 'iso' => 'TR', 'ip' => '109.185.255.0'],
        ['c' => '+87', 'n' => 'Kazakhstan', 'iso' => 'KZ', 'ip' => '164.0.255.0'],
        ['c' => '+852', 'n' => 'Hong Kong', 'iso' => 'HK', 'ip' => '103.115.40.0'],
        ['c' => '+82', 'n' => 'South Korea', 'iso' => 'KR', 'ip' => '106.247.55.0'],
        ['c' => '+7', 'n' => 'Kazakhstan', 'iso' => 'KZ', 'ip' => '164.0.255.0'],
        ['c' => '+65', 'n' => 'Singapore', 'iso' => 'SG', 'ip' => '69.192.164.0'],
        ['c' => '+64', 'n' => 'New Zealand', 'iso' => 'NZ', 'ip' => '222.154.13.0'],
        ['c' => '+61', 'n' => 'Australia', 'iso' => 'AU', 'ip' => '1.159.255.0'],
        ['c' => '+60', 'n' => 'Malaysia', 'iso' => 'MY', 'ip' => '60.48.0.0'],
        ['c' => '+49', 'n' => 'Germany', 'iso' => 'DE', 'ip' => '152.89.163.246'],
        ['c' => '+48', 'n' => 'Poland', 'iso' => 'PL', 'ip' => '103.112.60.0'],
        ['c' => '+47', 'n' => 'Norway', 'iso' => 'NO', 'ip' => '109.189.255.0'],
        ['c' => '+46', 'n' => 'Sweden', 'iso' => 'SE', 'ip' => '104.107.239.0'],
        ['c' => '+45', 'n' => 'Denmark', 'iso' => 'DK', 'ip' => '109.59.255.0'],
        ['c' => '+44', 'n' => 'United Kingdom', 'iso' => 'GB', 'ip' => '109.181.255.0'],
        ['c' => '+43', 'n' => 'Austria', 'iso' => 'AT', 'ip' => '128.131.255.0'],
        ['c' => '+423', 'n' => 'Liechtenstein', 'iso' => 'LI', 'ip' => '212.77.63.0'],
        ['c' => '+421', 'n' => 'Slovakia', 'iso' => 'SK', 'ip' => '147.175.255.0'],
        ['c' => '+420', 'n' => 'Czech Republic', 'iso' => 'CZ', 'ip' => '109.183.0.0'],
        ['c' => '+41', 'n' => 'Switzerland', 'iso' => 'CH', 'ip' => '129.194.0.0'],
        ['c' => '+40', 'n' => 'Romania', 'iso' => 'RO', 'ip' => '109.96.0.0'],
        ['c' => '+39', 'n' => 'Italy', 'iso' => 'IT', 'ip' => '101.59.255.0'],
        ['c' => '+389', 'n' => 'Macedonia', 'iso' => 'MK', 'ip' => '188.44.31.0'],
        ['c' => '+387', 'n' => 'Bosnia and Herzegovina', 'iso' => 'BA', 'ip' => '94.250.127.0'],
        ['c' => '+386', 'n' => 'Slovenia', 'iso' => 'SI', 'ip' => '109.182.255.0'],
        ['c' => '+385', 'n' => 'Croatia', 'iso' => 'HR', 'ip' => '161.53.171.0'],
        ['c' => '+382', 'n' => 'Montenegro', 'iso' => 'ME', 'ip' => '95.155.63.0'],
        ['c' => '+381', 'n' => 'Serbia', 'iso' => 'RS', 'ip' => '178.149.255.0'],
        ['c' => '+377', 'n' => 'Monaco', 'iso' => 'MC', 'ip' => '88.209.127.0'],
        ['c' => '+375', 'n' => 'Belarus', 'iso' => 'BY', 'ip' => '178.172.154.0'],
        ['c' => '+374', 'n' => 'Armenia', 'iso' => 'AM', 'ip' => '109.75.32.0'],
        ['c' => '+373', 'n' => 'Moldova', 'iso' => 'MD', 'ip' => '109.185.255.0'],
        ['c' => '+372', 'n' => 'Estonia', 'iso' => 'EE', 'ip' => '103.3.227.0'],
        ['c' => '+371', 'n' => 'Latvia', 'iso' => 'LV', 'ip' => '159.148.108.0'],
        ['c' => '+370', 'n' => 'Lithuania', 'iso' => 'LT', 'ip' => '158.129.255.0'],
        ['c' => '+36', 'n' => 'Hungary', 'iso' => 'HU', 'ip' => '134.255.127.0'],
        ['c' => '+359', 'n' => 'Bulgaria', 'iso' => 'BG', 'ip' => '109.121.255.0'],
        ['c' => '+358', 'n' => 'Finland', 'iso' => 'FI', 'ip' => '103.101.91.0'],
        ['c' => '+356', 'n' => 'Malta', 'iso' => 'MT', 'ip' => '141.8.127.0'],
        ['c' => '+354', 'n' => 'Iceland', 'iso' => 'IS', 'ip' => '103.199.98.0'],
        ['c' => '+353', 'n' => 'Ireland', 'iso' => 'IE', 'ip' => '101.44.255.0'],
        ['c' => '+352', 'n' => 'Luxembourg', 'iso' => 'LU', 'ip' => '103.178.29.0'],
        ['c' => '+351', 'n' => 'Portugal', 'iso' => 'PT', 'ip' => '104.132.255.0'],
        ['c' => '+34', 'n' => 'Spain', 'iso' => 'ES', 'ip' => '1.178.255.0'],
        ['c' => '+33', 'n' => 'France', 'iso' => 'FR', 'ip' => '1.179.127.0'],
        ['c' => '+32', 'n' => 'Belgium', 'iso' => 'BE', 'ip' => '103.109.244.0'],
        ['c' => '+31', 'n' => 'Netherlands', 'iso' => 'NL', 'ip' => '101.33.29.0'],
        ['c' => '+30', 'n' => 'Greece', 'iso' => 'GR', 'ip' => '102.38.248.0'],
        ['c' => '+1', 'n' => 'Canada', 'iso' => 'CA', 'ip' => '100.42.23.0'],
    ];

    $cleanPhone = preg_replace('/[^0-9]/', '', $phone);
    
    // Поиск соответствующего кода страны в массиве
    foreach ($countries as $country) {
      $cleanCountryCode = preg_replace('/[^0-9]/', '', $country["c"]);

      if (strpos($cleanPhone, $cleanCountryCode) === 0) { 
          return ['ip' => randomIpByCountry($country["iso"]), 'iso' => $country["iso"]];
      }
    }
    
    // Если не найдено, возвращаем 0.0.0.0
    return ['ip' => randomIpByCountry('DZ'), 'iso' => 'DZ'];
}

// Проверяем, установлен ли параметр phone в запросе
if (isset($_GET['phone'])) {
    echo getIp($_GET['phone']);
} else {
    echo "Please provide a phone number with the 'phone' parameter.";
}

?>
